/// <reference lib="webworker" />

// TypeScript declarations for service worker
declare const self: ServiceWorkerGlobalScope
export type {}

const CACHE_NAME = 'planner-v2'
const API_CACHE_NAME = 'api-cache-v1'
const FONT_CACHE_NAME = 'font-cache-v1'

// URLs to cache on install
const STATIC_ASSETS = [
  '/',
  '/index.html',
  '/manifest.json',
  '/offline.html'
]
// Note: font files are cached at runtime; if present, they will be cached on first load

// API endpoints to cache
const API_ROUTES = [
  '/api/planner/plans'
]

// Dev logging helper (disabled in production)
const isDev = typeof process !== 'undefined' && process.env.NODE_ENV !== 'production'
const logSW = (message: string, data?: any) => {
  if (isDev) {
    logger.log(`[ServiceWorker] ${message}`, data || '')
  }
}

/**
 * Install event - cache static assets
 */
self.addEventListener('install', (event: ExtendableEvent) => {
  logSW('Installing...')
  
  event.waitUntil(
    caches.open(CACHE_NAME).then((cache) => {
      logSW('Caching static assets')
      return cache.addAll(STATIC_ASSETS)
    }).then(() => {
      return self.skipWaiting()
    })
  )
})

/**
 * Activate event - clean old caches
 */
self.addEventListener('activate', (event: ExtendableEvent) => {
  logSW('Activating...')
  
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames.map((cacheName) => {
          if (cacheName !== CACHE_NAME && cacheName !== API_CACHE_NAME) {
            logSW('Deleting old cache:', cacheName)
            return caches.delete(cacheName)
          }
          return Promise.resolve()
        })
      )
    }).then(() => {
      return self.clients.claim()
    })
  )
})

/**
 * Fetch event - network-first with cache fallback
 */
self.addEventListener('fetch', (event: FetchEvent) => {
  const { request } = event
  const url = new URL(request.url)

  // Skip cross-origin requests
  if (url.origin !== self.location.origin) {
    return
  }

  // API requests - network-first strategy
  if (url.pathname.startsWith('/api/')) {
    event.respondWith(
      fetch(request)
        .then((response) => {
          // Cache successful GET requests
          if (request.method === 'GET' && response.ok) {
            const clone = response.clone()
            caches.open(API_CACHE_NAME).then((cache) => {
              cache.put(request, clone)
            })
          }
          return response
        })
        .catch(() => {
          // Fallback to cache on network failure
          return caches.match(request).then((cached) => {
            if (cached) {
              logSW('Serving from cache:', request.url)
              return cached
            }
            // Return offline page for navigation requests
            if (request.mode === 'navigate') {
              return caches.match('/offline.html').then((offline) => {
                return offline || new Response('Offline', { status: 503 })
              })
            }
            return new Response('Network error', {
              status: 503,
              statusText: 'Service Unavailable'
            })
          })
        })
    )
    return
  }

  // Font requests - cache-first strategy in dedicated cache
  if (url.pathname.startsWith('/fonts/') || request.headers.get('accept')?.includes('font')) {
    event.respondWith(
      caches.open(FONT_CACHE_NAME).then(async (cache) => {
        const cached = await cache.match(request)
        if (cached) return cached
        try {
          const response = await fetch(request)
          if (response.ok) cache.put(request, response.clone())
          return response
        } catch {
          // As a last resort, fall back to app cache or fail quietly
          const fallback = await caches.match(request)
          return fallback || new Response('', { status: 504 })
        }
      })
    )
    return
  }

  // Static assets - cache-first strategy
  event.respondWith(
    caches.match(request).then((cached) => {
      if (cached) {
        return cached
      }
      
      return fetch(request).then((response) => {
        // Cache successful responses
        if (response.ok) {
          const clone = response.clone()
          caches.open(CACHE_NAME).then((cache) => {
            cache.put(request, clone)
          })
        }
        return response
      })
    })
  )
})

/**
 * Background sync - process sync queue
 */
self.addEventListener('sync', (event: ExtendableEvent & { tag: string }) => {
  logSW('Background sync:', event.tag)
  
  if (event.tag === 'sync-planner') {
    event.waitUntil(
      // Notify app to process sync queue
      self.clients.matchAll().then((clients) => {
        return Promise.all(
          clients.map((client) =>
            client.postMessage({
              type: 'SYNC_REQUESTED',
              timestamp: Date.now()
            })
          )
        )
      })
    )
  }
})

/**
 * Message event - handle commands from app
 */
self.addEventListener('message', (event: ExtendableMessageEvent) => {
  logSW('Message received:', event.data)
  
  const { type, payload } = event.data

  switch (type) {
    case 'SKIP_WAITING':
      self.skipWaiting()
      break
      
    case 'CLEAR_CACHE':
      event.waitUntil(
        caches.keys().then((cacheNames) => {
          return Promise.all(
            cacheNames.map((cacheName) => caches.delete(cacheName))
          )
        }).then(() => {
          event.ports[0]?.postMessage({ success: true })
        })
      )
      break
      
    case 'CACHE_PLAN':
      if (payload?.plan) {
        event.waitUntil(
          caches.open(API_CACHE_NAME).then((cache) => {
            const request = new Request(`/api/planner/plans/${payload.plan.id}`)
            const response = new Response(JSON.stringify(payload.plan), {
              headers: { 'Content-Type': 'application/json' }
            })
            return cache.put(request, response)
          }).then(() => {
            event.ports[0]?.postMessage({ success: true })
          })
        )
      }
      break
  }
})

/**
 * Push event - show notification
 */
self.addEventListener('push', (event: PushEvent) => {
  logSW('Push notification received')
  
  const data = event.data?.json() || {}
  const title = data.title || 'MyRoadTrip'
  const options = {
    body: data.body || 'New update available',
    icon: '/icon-192.png',
    badge: '/badge-72.png',
    tag: data.tag || 'default',
    data: data.url
  }

  event.waitUntil(
    self.registration.showNotification(title, options)
  )
})

/**
 * Notification click - open app
 */
self.addEventListener('notificationclick', (event: NotificationEvent) => {
  logSW('Notification clicked')
  
  event.notification.close()

  event.waitUntil(
    self.clients.matchAll({ type: 'window', includeUncontrolled: true }).then((clients) => {
      // Focus existing window if available
      for (const client of clients) {
        if (client.url === '/' && 'focus' in client) {
          return client.focus()
        }
      }
      // Open new window
      if (self.clients.openWindow) {
        return self.clients.openWindow(event.notification.data || '/')
      }
      return Promise.resolve()
    })
  )
})

logSW('Loaded')
